const WebSocket = require('ws');
const http = require('http');
const fs = require('fs');
const path = require('path');

// === Configuration ===
const PORT = 45510;

// Récupère le dossier où se trouve le script
const baseDir = __dirname;
// Définit le dossier des logs à l'intérieur de ce répertoire
const logDir = path.join(baseDir, 'ChatLogs');

// Crée le dossier des logs s'il n'existe pas
if (!fs.existsSync(logDir)) {
    fs.mkdirSync(logDir, { recursive: true });
    console.log(`Dossier créé : ${logDir}`);
}

// Fonction pour écrire les logs
function writeLog(text) {
    const date = new Date();
    const dateStr = date.toISOString().split('T')[0]; // YYYY-MM-DD
    const logFile = path.join(logDir, `${dateStr}.log`);
    fs.appendFileSync(logFile, `[${date.toISOString()}] ${text}\n`);
}

// === Serveur WebSocket ===
const server = http.createServer();
const wss = new WebSocket.Server({ server });

const clients = new Map(); // ws => username

wss.on('connection', (ws, req) => {
    const ip = req.socket.remoteAddress;

    ws.on('message', (message) => {
        try {
            const data = JSON.parse(message);

            if (data.type === 'setUsername') {
                const nameTaken = Array.from(clients.values()).includes(data.username);
                if (nameTaken) {
                    ws.send(JSON.stringify({ type: 'usernameStatus', status: 'taken' }));
                } else {
                    clients.set(ws, data.username);
                    ws.send(JSON.stringify({ type: 'usernameStatus', status: 'ok' }));
                    writeLog(`IP ${ip} enregistré en tant que "${data.username}"`);
                }
            } else if (data.type === 'message') {
                const username = clients.get(ws) || 'Inconnu';
                const fullMessage = `${username}: ${data.message}`;
                writeLog(fullMessage);
                // Broadcast à tous les clients
                wss.clients.forEach((client) => {
                    if (client.readyState === WebSocket.OPEN) {
                        client.send(JSON.stringify({ type: 'chat', message: fullMessage }));
                    }
                });
            }
        } catch (err) {
            console.error('Erreur traitement message:', err);
        }
    });

    ws.on('close', () => {
        const username = clients.get(ws);
        if (username) {
            clients.delete(ws);
            writeLog(`Déconnexion de "${username}"`);
        }
    });
});

server.listen(PORT, () => {
    console.log(`Serveur WebSocket démarré sur le port ${PORT}`);
});
